<?php
/* 
 * Ce module permet d'afficher les logos de certaines des catégories d'un 
 * produit qui auront été configurées à l'installation du module.
 * 
 * L'utilisation typique est d'afficher les labels d'un produit dans un onglet
 * spécifique de la page produit. 
 *
 * Utilise les HOOK standards HOOK_PRODUCT_TAB et HOOK_PRODUCT_TAB_CONTENT
 * 
 * A noter, pour positionner l'onglet, il faut afficher les hook non visibles et naviguer
 * jusqu'au hook product tab et placer judicieusement ce dernier : le premier arrivé est le premier servi
 *
 * La configuration du module permet de choisir les catégories à afficher.
 * Cette information est stockée en bdd dans la configuration (ps_configuration) - NN_CATEGORIESFORPRODUCT_CATS
 *
 */

if (!defined('_PS_VERSION_'))
	exit;
	
class nnCategoriesForProduct extends Module
{
    
    public function __construct()
    {
        $this->name = 'nncategoriesforproduct';
        $this->tab = 'front_office_features';
        $this->version = 1.0;
        $this->author = 'Noir Ô Naturel';
        $this->need_instance = 0;
        // Override for development
        $this->is_configurable = 1;
        
        parent::__construct();
        
        $this->page = basename(__FILE__, '.php');
        $this->displayName = $this->l('N&N : Categories for Product');
        $this->description = $this->l('Display selected category logos in product page');

        $this->_initialized = false;
    }
    
    /**
     * Module installation (i.e. clic on install button in administration/mmodule tab
     */
    function install()
    {
        if (
          /* Parent installation */
          !parent::install() 
        
          /* Register hook for display on product page */
          || !$this->registerHook('displayProduct_3')
          || !$this->registerHook('header')
                  
          /* Set configuration value listing category names to display */
          || !Configuration::updateValue('NN_CATEGORIESFORPRODUCT_CATS', '')
        ) 
        return false;
    
        return true;
    }
    
    /**
     * Called when deinstalling the module, reverse installation
     */     
    public function uninstall()
    {
        if (  
          !Configuration::deleteByName('NN_CATEGORIESFORPRODUCT_CATS')
          || !parent::uninstall()
        ) 
        return false;
    
        return true;
    }

    /**
     * Initialized data
     */
    private function initContent() 
    {
        global $cookie;
        global $smarty;

        if ( $this->_initialized ) 
          return;

        $id_product = (int)Tools::getValue('id_product');
        $id_lang = (int)$cookie->id_lang;
          
        if (!Validate::isUnsignedId($id_product))
          die(Tools::displayError());
      
        /* Get all categories for id_product and relative name and url */
        $categories = Product::getProductCategoriesFull($id_product,$id_lang);

        /* Get category ids to display */
        $items = Configuration::get('NN_CATEGORIESFORPRODUCT_CATS');
        $items = explode(',', $items);
        $catsToDisplay=array();
        foreach($items as $item)
        {
            $id = (int)substr($item, 3, strlen($item));
            $catsToDisplay[$id]=true;
        }
        
        /* Keep only categories to display */
        $categories = array_intersect_key($categories, $catsToDisplay);

        /* Set id image as expected see Category.php class for details */
        foreach ($categories as &$cat)
        {            
            $cat['id_image'] = (file_exists(_PS_CAT_IMG_DIR_.$cat['id_category'].'.jpg')) ? (int)($cat['id_category']) : Language::getIsoById($id_lang).'-default';
            $cat['legend'] = 'no picture';
        }        

        /* Assign to smarty */
        $smarty->assign('nb_categories',0);
        if (!empty($categories)) 
        {
            $smarty->assign(array(
                'nb_categories' => count($categories),
                'categories' => $categories,
                'mediumSize' => Image::getSize('medium'),
                'smallSize' => Image::getSize('small')
            ));
        }

        $this->_initialized = true;
    }
    
    /**
     * Hook call
     */     
    public function hookDisplayProduct_3($params)
    {        
        global $smarty;
        $this->initContent();
        $nb_categories = (int)$smarty->getTemplateVars('nb_categories');
        if ($nb_categories != 0)
            return ($this->display(__FILE__, '/nnCategoriesForProduct.tpl'));
        else
            return null; 
    }

    /**
     * Build recursively a list of categories
     * returns the categories as a list of option starting from Home     
     */
    private function _getCategoryOptionRec($id_category, $id_lang, $children = true)
    {        
        $categorie = new Category($id_category, $id_lang);
        if(is_null($categorie->id)) return '';
        if(count(explode('.', $categorie->name)) > 1)
          $name = str_replace('.', '', strstr($categorie->name, '.'));
        else
          $name = $categorie->name;
        $res = '<option value="CAT'.$categorie->id.'" style="margin-left:'.(($children) ? round(15+(15*(int)$categorie->level_depth)) : 0).'px;">'.$name.'</option>';
        if($children)
        {
          $childrens = Category::getChildren($id_category, $id_lang);
          if(count($childrens))
            foreach($childrens as $children)
              $res .= $this->_getCategoryOptionRec($children['id_category'], $id_lang);
        }
        return $res;
    }


    /**
     *  Returns the administrative form
     */
    public function getContent()
    {
        global $cookie;
        
        // In reaction to save
        if(Tools::isSubmit('submitNNCategoriesForProduct'))
        {
          if(Configuration::updateValue('NN_CATEGORIESFORPRODUCT_CATS', Tools::getValue('items')))
            $this->_html .= $this->displayConfirmation($this->l('Settings Updated'));
          else
            $this->_html .= $this->displayError($this->l('Unable to update settings'));
        }

        $items_str = Configuration::get('NN_CATEGORIESFORPRODUCT_CATS');
        $items = explode(',', $items_str);
                  
        // Administration form
        $this->_html .= '
        <fieldset>
            <legend><img src="'.$this->_path.'logo.gif" alt="" title="" />'.$this->l('Categories to display').'</legend>
            <form method="post" action="'.Tools::safeOutput($_SERVER['REQUEST_URI']).' id="form">        
                <div style="display: none">
                   <label>'.$this->l('Categories').'</label>
                   <div class="margin-form">
                        <input type="text" name="items" id="itemsInput" value="'.$items_str.'" size="70" />
                   </div>
                </div>
                <div class="clear">&nbsp;</div>
                    <table style="margin-left: 130px;">
                        <tbody>
                            <tr>
                                <td>
                                    <select multiple="multiple" id="items" style="width: 300px; height: 160px;">';
                                    foreach($items as $item)
                                    {
                                        $id = (int)substr($item, 3, strlen($item));
                                        $this->_html .= $this->_getCategoryOptionRec($id, $cookie->id_lang, false);
                                    }
                                    $this->_html .= '</select>
                                    <br />
                                    <br />
            			            <a href="#" id="removeItem" style="border: 1px solid rgb(170, 170, 170); margin: 2px; padding: 2px; text-align: center; display: block; text-decoration: none; background-color: rgb(250, 250, 250); color: rgb(18, 52, 86);">'.$this->l('Remove').' &gt;&gt;</a>
                                </td>
                                <td style="padding-left: 20px;">
                                    <select multiple="multiple" id="availableItems" style="width: 300px; height: 160px;">
                                    ';
                                        $this->_html .= '<optgroup label="'.$this->l('Categories').'">';                                        
                                        $this->_html .= $this->_getCategoryOptionRec(1, $cookie->id_lang);
                                        $this->_html .= '</optgroup>';
                                        $this->_html .= '
                                    </select><br />
                                    <br />
                                    <a href="#" id="addItem" style="border: 1px solid rgb(170, 170, 170); margin: 2px; padding: 2px; text-align: center; display: block; text-decoration: none; background-color: rgb(250, 250, 250); color: rgb(18, 52, 86);">&lt;&lt; '.$this->l('Add').'</a>			
                                </td>
                            </tr>
                        </tbody>
                    </table>
                <div class="clear">&nbsp;</div>
                <script type="text/javascript">
                    $(document).ready(function() {
                      $("#addItem").click(add);
                      $("#availableItems").dblclick(add);
                      $("#removeItem").click(remove);
                      $("#items").dblclick(remove);
                      function add()
                      {
                        $("#availableItems option:selected").each(function(i){
                          var val = $(this).val();
                          var text = $(this).text();
                          $("#items").append("<option value=\""+val+"\">"+text+"</option>");
                        });
                        serialize();
                        return false;
                      }
                      function remove()
                      {
                        $("#items option:selected").each(function(i){
                          $(this).remove();
                        });
                        serialize();
                        return false;
                      }
                      function serialize()
                      {
                        var options = "";
                        $("#items option").each(function(i){
                          options += $(this).val()+",";
                        });
                        $("#itemsInput").val(options.substr(0, options.length - 1));
                      }
                    });
                </script>
                <p class="center">
                  <input type="submit" name="submitNNCategoriesForProduct" value="'.$this->l('  Save  ').'" class="button" />
                </p>
            </form>
        </fieldset><br />';    
        return $this->_html;
    }
    
    /**
     *  Definit la feuille de style
     */
    function hookHeader($params)
	{
	   $this->context->controller->addCSS(($this->_path).'css/default.css', 'all');
	}

}
